"""tablcgi.py
   Classes to provide support for simple database tables for
   MRS and CGI applications with semi-persistent data.

   To make this data persistent, file locking is provided,
   by using the LockFile module copied from MailMan Python 
   source code, see www.list.org .

   table_def class is used to create multiple table definition
   objects based on independent (cloned) data so that more than
   one table of a particular kind can be opened at the
   same time without cross referring to (and messing with) each 
   others attribute values, e.g. filename.

   table class contains methods to load, save, print as HTML,
   and to find, add, remove and modify rows. Objects store a 
   list of dictionaries, each dictionary is one row. Objects also
   store meta data, used to control column operations,
   and the filename in which the updated object is 
   saved as a pickle file. 

   See tester() code for usage examples.
   and read the source code for the full documentation.
   
   Distributed under the terms of the GNU Public License, 
   version 3 or higher, see http://www.gnu.org/ for details.

   This program comes without warranty of any kind.

   Richard Kay, 26/04/02"""


class ColumnException(Exception):
  def __init__(self,err=""):
      self.value = "tablcgi.py: column error: "+err
  def __str__(self):
    return repr(self.value)

class RowException(Exception):
  def __init__(self,err=""):
      self.value = "tablcgi.py row error: " +err
  def __str__(self):
    return repr(self.value)


# default file access permissions when a new database file is created 
# see chmod(1)
# a mode of 640 corresponds to rw-r-----
default_file_creation_mode='640'

class update_set:
  ''' builds a list of code references and parameters for updates which
  are to be grouped together. dispatch method executes these
  all in one go. 

  TBD - make what this does fully atomic with a rollback if it doesn't 
  complete and a commit if it does.  Possible
  way forward is store an object which for each atomic update set:
  a. backs up all data to be modified/removed.
  b. monitors status of transaction - this needs to be carried
  out prior to starting any update to check all resources are 
  ready, periodically and on system startup to check for and either 
  roll back or clean up any committed but incomplete transactions.
  c. says what its going to do, with new data.
  d. records status: Preparation -> Processing -> Commit -> Complete
  
  Preparation includes taking backups, locking all resources.
  Processing means updating the target files and flushing to disk.
  Commit means passing point of no return, removing locks and cleaning 
  up rollback resources.
  Complete means everything cleaned up.

  Rollback is what happens when monitoring detects an update having
  entered the processing state and having crashed or hung prior to 
  commit.
  '''

  def __init__(self,logging=0,logfile='MRS_update_together.log'):
    ''' initialises an update_set object '''
    self.logfile=logfile
    self.logging=logging
    self.exec_list=[]
    if self.logging: 
      self.lfp=open(logfile,'a')
  def add_entry(self,coderef,params):
    ''' takes code reference and parameters and adds entry to
    the execution list'''
    entry={'coderef':coderef,'params':params}
    self.exec_list.append(entry)
    if self.logging: 
      self.lfp.write('update_together.add_entry: '+ repr(entry)+'\n')
  def dispatch(self):
    ''' executes all entries in the execution list in one go.'''
    for entry in self.exec_list:
      if self.logging: 
        self.lfp.write('update_together.dispatch: '+ repr(entry)+'\n')
      apply(entry['coderef'],entry['params'])


class table_def:
  ''' clones a specialised table_def object based on a table_def class
  in other modules, allowing multiple table definitions derived from 
  the same original to coexist without clobbering each others attributes.
  This is achieved using the [:] full slice operator, which does a
  full copy of a list-type item (string,list,or tuple). This approach 
  doesn't just provide another reference to the same object. A new 
  meta-data dictionary is built from scratch using cloned strings.'''
  def __init__(self,defobj):
    if hasattr(defobj,"sqfile"): # will for sq_table, wont for base class
      if defobj.sqfile:
        self.sqfile=defobj.sqfile[:] # use full slice to clone string
      else:
        self.sqfile=None
    if defobj.file: 
      self.file=defobj.file[:] # use full slice to clone string
    else:
      self.file=None
    self.index_col=defobj.index_col[:]
    self.meta={}
    for key in defobj.meta.keys():
      selfkey=key[:] # clone key
      self.meta[selfkey]=defobj.meta[key][:] # clone list indexed by key

class table:
  """ class which stores a table using a pickle file and do
  add/mod/del row operations on the table. table data is a list of
  dictionaries each dictionary is for a row. meta data is a dictionary 
  of lists each list holding ordered information about columns for
  display purposes """
  def __init__(self,defin,dir=None,read_locked=False,save_asap=1):
    ''' table defin(ition) has attributes meta,file,index_col
     
    if read_locked == True the table will be locked prior to
    opening in order to maintain the lock for the duration of
    table object open->read->close->open->write->close cycle
    which may be required to prevent concurrent updates between 
    connected read and write operations on same table object.
    
    if self.filename has value None, the table is a view and is neither
    loaded from nor saved to a file. 
    
    in order to prevent unwanted manipulation of attributes through
    the defin parameter object, first clone it.
    save_asap controls whether or not to save ASAP after table row
    add/mod/del operations.
    '''
    self.defin=table_def(defin)
    self.meta=self.defin.meta
    # meta is information about data, is a set of lists in matching order.
    # each element of a list is information about a column (col) in data.
    # meta["keys"] = list of keys in the column order of printing
    # meta["colheads"] = column headings for printing
    # meta["formats"] = print formats e.g. '%.2f', '%s', '%d'
    # 
    # dir allows multiple instances of the table in different
    # directories. dir is the directory prefix of the file.
    if dir and defin.file:
      import os
      self.file=os.path.join(dir,self.defin.file)
      self.lockfilename=self.file+'.lock'
    elif defin.file:
      self.file=self.defin.file
      self.lockfilename=self.file+'.lock'
    else:
      self.file=None
      self.lockfilename=None
    # file is where the filename data is loaded from and saved to
    self.read_locked=read_locked
    self.save_asap=save_asap
    self.index_col=self.defin.index_col
    if not self.index_col in self.meta["keys"]:
      raise ColumnException(err=" index column metadata error")
    # index_col is the column used to index data
    # data is a list of dictionaries saved in filename
    self.data=self.load()
  
  def load(self): # loads data from file, or empty list
    """ loads serialised data object from pickle file """
    import pickle, LockFile
    data = [] # default if can't read it
    if not self.file:
      return data # a memory constructed view is neither loaded nor saved
    if self.read_locked: 
      self.lock=LockFile.LockFile(self.lockfilename,withlogging=1)
    try:
      if self.read_locked: 
        try:
          self.lock.lock(15)
          fp=open(self.file,"rb")
          data=pickle.load(fp)
          fp.close()
          self.lock.unlock()
        except LockFile.LockError:
          self.lock.unlock()
          raise LockFile.LockError
      else:
        fp=open(self.file,"rb")
        data=pickle.load(fp)
        fp.close()
      return data
    except(IOError,EOFError): # at start no rows in table
      return data
  
  def save(self):
    """ saves structured object into pickle file. Uses Mailman 
    LockFile() module. returns 1 if update successful, otherwise 0.
    When adding or deleting a row there is no need to lock
    before reading data. However, when modifying a row, in
    order to prevent race conditions between processes updating
    different columns in the same row, it is neccessary
    to lock the row before current contents are read, so that
    updates which have occurred between the current processes' 
    read and write operations are not overridden. 
    """
    if not self.file:
      return 0 # a memory constructed view is neither loaded nor saved
    import pickle
    import LockFile
    import os,os.path
    if not os.path.isfile(self.file):
      newfile=True
    else:
      newfile=False
    if not self.read_locked: 
      self.lock=LockFile.LockFile(self.lockfilename,withlogging=1)
    try:
      try:
        if not self.read_locked: 
          self.lock.lock(15)
        fp=open(self.file,"wb")
        pickle.dump(self.data,fp)
        fp.flush()
        fp.flush()
        fp.close()
      finally:   
        self.lock.unlock()
    except LockFile.LockError:
      return 0 # didn't update stored table
    else:
      if newfile:
        chmod_cmd='chmod '+default_file_creation_mode+' '+self.file
        os.system(chmod_cmd)
      return 1 # did update stored table
  
  def find(self,value):
    """ returns index of 1st record whose index_col equals value 
        or returns -1"""
    if not self.index_col in self.meta["keys"]:
      raise ColumnException
    i=0
    for rec in self.data:
      if rec[self.index_col] == value:
        return i
      i+=1
    return -1 # value not present in column
  
  def has_key(self,value):
    """ returns 1 if value present within index column or 0 """
    if self.find(value) == -1:
      return 0
    else:
      return 1
  
  def keys(self):
    """ returns index column as a list """
    if not self.index_col in self.meta["keys"]:
      raise ColumnException
    keylist=[]
    for rec in self.data:
       keylist.append(rec[self.index_col])
    return keylist
  
  def addrow(self,row):
    """ adds row which must be
        validated before calling this method.  """

    keys=self.meta["keys"][:] # need a copy of keys list, not another
    # reference to it or sort will put meta["keys"] into wrong order !!
    rowkeys=list(row.keys())
    keys.sort()
    rowkeys.sort()
    if ((not keys == rowkeys) or 
        (self.index_col and not self.index_col in keys)):
      print( "metakeys: ",keys," row: ",rowkeys)
      raise ColumnException(err="table.addrow: key mismatch")
    if self.index_col:
      atrow=self.find(row[self.index_col])
      if atrow != -1:
        raise RowException(err="table.addrow: row already exists") 
    self.data.append(row) # add row at end of data
    if self.save_asap :
      return self.save() # returns 0 or 1
    else:
      return -1
  
  def modrow(self,row):
    ''' Re-reads data and modifies fields present in
    row before saving. Leaves fields not present in row
    unchanged.
    Parameters: 
    row - dictionary containing unique column plus 1 or more 
      key/value sets of field updates for row to be modified. Must 
      include unique column value for row to be updated, and fields
      to be modified. row must not include fields to be left alone.
    '''
    
    # first do some sanity checking
    if type(row) != type({}):
      raise TypeError
    if not self.index_col in list(row.keys()):
      raise ColumnException
    row_key=row[self.index_col]
    row_index=self.find(row_key)
    if row_index == -1:
      raise RowException # cant mod row that doesn't exist
    for key in row.keys():
      if key not in self.meta['keys']: # extraneous key
        raise ColumnException 
      if key != self.index_col:
        self.data[row_index][key]=row[key]
    if self.save_asap :
      return self.save() # returns 0 or 1 
    else:
      return -1
  
  def delrow(self,value):
    """ deletes row. index_col required and row with same column value must
        already exist or row will not be deleted """
    keys=self.meta["keys"]
    if not self.index_col in keys:
      raise ColumnException
    atrow=self.find(value)
    if atrow == -1:
      raise RowException # cant del row that doesn't exist
    del self.data[atrow] # remove row whose col == value
    if self.save_asap :
      return self.save() # returns 0 or 1
    else:
      return -1

  def sort(self,key=None,ascending=True,cmpfunc=None):
    ''' sorts the table '''
    keys=self.meta["keys"]
    if not key and not cmpfunc and not self.index_col in keys:
      raise ColumnException
    if key and key not in keys:
      raise ColumnException
    if not key and not cmpfunc:
      key=self.index_col
    # define sort comparison function
    def dummy(): # do nothing function for type check  
      pass
    if not cmpfunc:
      def cmpfunc(row1,row2):
          if row1[key] > row2[key]:
              return 1
          elif row1[key] < row2[key]:
              return -1
          else:
              return 0
    elif type(cmpfunc) != type(dummy):
      raise ValueError
    # do the sort
    from functools import cmp_to_key
    self.data.sort(key=cmp_to_key(cmpfunc),reverse=(not ascending))
    self.save()
 
  def tab2html(self,skip_cols=[],align='center',
               bgcolor='#ffffff',border=2,
               color_row_key=None):
    ''' 
    Prints data as a HTML table using meta data for output formatting 
    
    Parameters 
    skip_cols: list of keys which are not printed
    align: a valid html align tag
    bgcolor: this applies to entire table background color
    border: HTML table border width
    color_row_key: if present, name of a key with a bgcolor
                   value applicable on a row by row basis. This key
                   should be in skip_cols, its value used as row bgcolor.
    '''
    printstr= "<p>\n"
    printstr+= """<table align="%s" 
                    bgcolor="%s" 
                    border=%d >\n""" % (align,bgcolor,border)
    # print column headings
    printstr+= "<tr>\n"
    i=0
    for heading in self.meta["colheads"]:
      key=self.meta["keys"][i]
      if key not in skip_cols: 
        printstr+= "<th> %s </th>\n" % heading
      i+=1
    printstr+= "</tr>\n"
    # loop through rows
    for row in self.data:
      if not color_row_key:
        printstr+= "<tr>\n"
      else:
        printstr+= "<tr bgcolor="+row[color_row_key]+" >\n"
      # loop through cols
      i=0      
      for key in self.meta["keys"]:
        # format object can be a printf conversion e.g. "%.2f" or
        # it could be a code reference or None
        format_obj=self.meta["formats"][i]
        is_string=0 
        is_code=0
        def dummy_code_obj(): pass # dummy code object for code type
        if type(format_obj) == type(""):
          is_string=1 
          formatstr="<td> %s </td>\n" %  format_obj
        elif type(format_obj) == type(dummy_code_obj):
          is_code=1
        # substitute %s|%f|%d type format conversion
        if key not in skip_cols and is_string:
          printstr+= formatstr % row[key] # substitute value
        elif key not in skip_cols and is_code:
          printstr+= "<td>\n"
          printstr+= format_obj(row[key]) # call code object to return string value
          printstr+= "</td>\n"
        i+=1
      printstr+= "</tr>\n"
    printstr+= "</table>\n"
    printstr+= "</p><br>\n"
    print(printstr)

class sq_table(table):
  ''' Sequenced table where the index column is a numeric
  sequence generated as a surrogate key. This inherits from table class.
  Before using this class ensure the table_def object has a
  sqfile attribute giving the filename for the pickle storing
  the sequence number. This attribute may have the value: None
  if the sequenced table is a memory only i.e. non persistent
  object. '''

  def __init__(self,defin,dir=None,read_locked=False,save_asap=1):
    ''' overrides base class constructor. Initialises 
    the sequence file if it needs to be newly created. '''

    import os.path
    # call base class contructor
    table.__init__(self,defin,
                   dir=dir,
                   read_locked=read_locked,
                   save_asap=save_asap)
    if self.file:
      if not hasattr(defin,"sqfile"):
        errmes="persistent sq_table definition requires a sequence filename"
        raise AttributeError( errmes)
      if dir:
        import os
        # prepend directory and clone it so we don't just copy a ref
        self.sqfile=os.path.join(dir,defin.sqfile[:])
      else: 
        self.sqfile=defin.sqfile[:] # clone it so we don't just copy a ref
    if not self.file:
      # start with 1 when we construct a memory only non-persistent table.
      self.nextnum=1
    elif not os.path.isfile(self.sqfile):
      # create the sequence number file if it doesnt exist and is needed 
      import pickle
      import LockFile
      lockfilename=self.sqfile+".lock"
      lock=LockFile.LockFile(lockfilename,withlogging=1)
      try:
        try:
          lock.lock(15)
          wfh=open(self.sqfile,"wb")
          pickle.dump(1,wfh)
          self.nextnum=1
          wfh.close()
        finally:
          lock.unlock()
      except LockFile.LockError:
        errormes="could not get exclusive lock on "+ self.sqfile
        raise( LockFile.LockError, errormes)
      chmod_cmd='chmod '+default_file_creation_mode+' '+self.sqfile
      os.system(chmod_cmd)
    else:
      # we must be reinitialising a persistent sequence table 
      self.nextnum=self.get_nextnum() # for memory storage

  def addrow(self,row):
    ''' overrides base class addrow method for situations where sequence
    number is computed '''
    if self.index_col in list(row.keys()): # base class addrow method should work
      table.addrow(self,row)
    else:
      # need to obtain next sequence value for index_col
      row[self.index_col]=self.get_nextnum() # get next number
      table.addrow(self,row)
    self.set_nextnum() # cause stored sequence number to increment

  def get_nextnum(self): 
    ''' Obtains the next number to be allocated for the next row to
    be added. A client can ask using this method before calling
    the addrow() method if it needs to know. '''   
    if not self.file:
      if not hasattr(self,"nextnum"):
        raise AttributeError ( "memory only table with no initial nextnum")
      # must be a memory only non-persistent table.
      return self.nextnum
    if not hasattr(self,"sqfile"):
      raise AttributeError ( "persistent sq_table class requires a sequence filename")
    import pickle
    import LockFile
    lockfilename=self.sqfile+".lock"
    lock=LockFile.LockFile(lockfilename,withlogging=1)
    try:
      try:
        lock.lock(15)
        try:
          sfh=open(self.sqfile,"rb")
          self.nextnum=pickle.load(sfh)
          sfh.close()
        except (EOFError,IOError):
          return 0 # if sequence file doesn't exist we have to start somewhere
      finally:
        lock.unlock()
    except LockFile.LockError:
      errormes="could not get exclusive lock on "+ self.sqfile
      raise LockFile.LockError( errormes)
    return self.nextnum
 
  def set_nextnum(self):     
    ''' causes the stored next number to be incremented '''
  
    if not self.file:
      if not hasattr(self,"nextnum"):
        raise AttributeError( "sq_table class requires a nextnum sequence number")
      # must be a memory only non-persistent table.
      self.nextnum+=1
      return # no more to do.
    if not hasattr(self,"sqfile"):
      raise AttributeError( "persistent sq_table class requires a sequence filename")
    import pickle
    import LockFile
    lockfilename=self.sqfile+".lock"
    lock=LockFile.LockFile(lockfilename,withlogging=1)
    try:
      try:
        lock.lock(15)
        try:
          rfh=open(self.sqfile,"rb")
          nextnum=pickle.load(rfh)
          rfh.close()
        except (IOError,EOFError):
          # if file doesn't exist, we have to start somehow
          self.nextnum=0
        wfh=open(self.sqfile,"wb")
        pickle.dump(nextnum+1,wfh)
        self.nextnum=nextnum+1
        wfh.close()
      finally:
        lock.unlock()
    except LockFile.LockError:
      errormes="set_nextnum: could not get exclusive lock on "+ self.sqfile
      raise LockFile.LockError( errormes)
 
  def sync_sqfile(self):     
    ''' provided for situation where data is imported from
    elsewhere without the sqfile being present.
    Use this method as follows:
    
    a. Substitute table_defintion_object for table to be sync_ed eg. 
    use atab for acknowledgements table 
     
    b. 
    from table_details import table_definition_object 
    import tablcgi
    tabl=tablcgi.sq_table(table_definition_object)
    tabl.sync_sqfile()
    '''
    
    if not self.file:
      raise AttributeError("sq_table.sync_sqfile requires a file")
    keys=self.keys()
    topfound=0
    for key in keys:
      if key > topfound:
        topfound=key
    nextnum=topfound+1
    import pickle
    import LockFile
    lockfilename=self.sqfile+".lock"
    lock=LockFile.LockFile(lockfilename,withlogging=1)
    try:
      try:
        lock.lock(15)
        wfh=open(self.sqfile,"wb")
        pickle.dump(nextnum,wfh)
        self.nextnum=nextnum
        wfh.close()
      finally:
        lock.unlock()
    except LockFile.LockError:
      errormes="sync_sqfile: could not get exclusive lock on "+ self.sqfile
      raise LockFile.LockError( errormes)

def tester():
  """ Runs tests on table class. Each use of the menu is intended to
      create valid HTML output which can be cut/pasted/saved using 
      an HTML source editor e.g. Quanta plus. HTML validation/viewing 
      can be done using HTML development environment. The table class
      will create a pickle file: temp.pkl and use this to store data 
      between program runs. """
  def hash_mark(mark):
    # demonstrates use of passed code as formatting object
    return "## %.1f ##" % mark
  
  class tabledefinition:
    pass 
  defin=tabledefinition()
  # meta data and definitions for test table
  defin.meta={"keys":["snum","name","mark"],
        "colheads":["Student No.","Name","Mark"],
        "formats":["%s","%s",hash_mark]}
  defin.index_col="snum"
  defin.file="temp.pkl"
  recs=[{"snum":"12345670","name":"Peter Smith"   ,"mark":45.2},
        {"snum":"01234567","name":"Wu Chan"       ,"mark":68.7},
        {"snum":"80123456","name":"Siobhan Murphy","mark":59.2}]
 
  # meta data and definitions for test sq_table
  sqdefin=tabledefinition()
  # similar to defin but needs an extra column for the sequence number
  sqdefin.meta={"keys":["sq","snum","name","mark"],
        "colheads":["Seq","Student No.","Name","Mark"],
        "formats":["%d","%s","%s",hash_mark]}
  sqdefin.index_col="sq"
  sqdefin.file="tempsq.pkl"
  # sqdefin needs an extra filename attribute for the sequence file
  sqdefin.sqfile="tempsq_sqk.pkl"
  xkeysadded=[] # stored keys to be added for sq_table test session
  # These start the same as the base class test records, but adding
  # sequence column values using references changes this data in
  # a manner incompatible with the base class tests. 
  sqrecs=[{"snum":"12345670","name":"Peter Smith"   ,"mark":45.2},
        {"snum":"01234567","name":"Wu Chan"       ,"mark":68.7},
        {"snum":"80123456","name":"Siobhan Murphy","mark":59.2}]
  xtbl=None # no extended table to start with
  persistent = -1 # meaning not yet set 
  readlock=int(input('enter 1 for read/write locking or 0 write locking'))
  if readlock:
    readlock=True
  else:
    readlock=False
  while 1: 
    # instantiate table object with no records (first time round)
    printstr= "<pre>" # menu works interactively and in a html
    printstr+= "enter   to test\n"
    printstr+= "  1     add record\n"
    printstr+= "  2     modify record\n"
    printstr+= "  3     delete record\n"
    printstr+= "  4     show table as HTML\n"
    printstr+= "  5     sort records\n"
    printstr+= "  6     adds records to sq_table class\n"
    printstr+= "  7     remove all sq_table records\n"
    printstr+= "  q     Quit\n"
    print(printstr)
    test=input("option: ")
    print( "</pre>")
    if test == '1':
      print( "test 1 recs: "+str(recs)) # debug
      recno=int(input("which record to add (0-2): "))
      tbl=table(defin,read_locked=readlock)
      tbl.addrow(recs[recno]) 
      # cgiutils2.html_end()
    elif test == '2':
      recno=int(input("which record to change (0-2): "))
      mark=float(input("enter new mark (0.0-100.0): "))
      recs[recno]["mark"]=mark
      tbl=table(defin,read_locked=readlock)
      tbl.modrow(recs[recno])
      # cgiutils2.html_end()
    elif test == '3':
      recno=int(input("which record to delete (0-2): "))
      tbl=table(defin,read_locked=readlock)
      tbl.delrow(recs[recno]["snum"])
      # cgiutils2.html_end()
    elif test == '4':
      tbl=table(defin) # don't lock if only reading not writing
      tbl.tab2html()
      # cgiutils2.html_end()
    elif test == '5':
      key=input("which sort key ( snum | name | mark )")
      asc=input("want ascending order ? y/n") 
      tbl=table(defin,read_locked=readlock)
      if asc == 'y' or asc == 'Y':
        tbl.sort(key)
      else:
        tbl.sort(key,ascending=False) 
    elif test == '6':
      if persistent == -1:
        persistent=int(input("enter 1 for saved table in /tmp or 0 for memory only "))
      if not persistent: 
        sqdefin.file=None
        sqdefin.sqfile=None
      # dont reuse the tbl ref, use extended object ref xtbl. When I
      # reused the base class reference this resulted in weird bugs
      if not xtbl: 
        xtbl=sq_table(sqdefin,dir='/tmp',read_locked=readlock)
      for row in sqrecs:
        # first store the next key for the row about to be added
        xkeysadded.append(xtbl.get_nextnum())
        # add the row
        xtbl.addrow(row)
      xtbl.tab2html()
    elif test == '7':
      if not xkeysadded:
        print("no rows to delete")
      if not xtbl: 
        xtbl=sq_table(sqdefin,dir='/tmp',read_locked=readlock)
      print( "before row deletion")
      xtbl.tab2html()
      # remove contents of sequence table
      for key in xkeysadded:
        xtbl.delrow(key)
      print( "after row deletion")
      xtbl.tab2html()

    elif test.lower() == 'q':
      # cgiutils2.html_end()
      break
    else:
      print( "invalid test option: "+test+'\n')
  
if __name__ == "__main__": # unit test condition
  tester()
