#!/usr/bin/env python

# uncomment next 2 lines to debug script
#import sys
#sys.stderr=sys.stdout

''' list_configure.py Richard Kay  Dec 2023
    This script is used to modify configuration variables
    needed for a new nextlist email list, or to change a list configuration. 
'''

import sys
import os,os.path,pickle
import shutil
import cgiutils

class md:
  ''' module data namespace class
  # md.cv contains data useful in assisting setting
  # of configured site variables likely to need changing.
  # 
  # Each file requires entries for filname, keys and information
  # for each variable as below.
  #  
  # Each variable requires:
  # 'default': A default value 
  # 'prompt': a user prompt to change the value 
  # 'help': further helpful information for the user
  '''
  # configuration files listed here so these can be processed in this order 
  cfgflist=['site_vars'] 
  cv={  

      "site_vars":{
        "filename":"list.cfg", 
        "keys":['obscure_email','owner_email','UID','GID','domain','webroot',
                'WhoCanPost',
               'UnsubscribeMethod','Visibility','MaxLength','DownloadDays'], 
        "vars":{

      'obscure_email':{
      'default':'arandompassword',
      'prompt':'obscure valid local site email address prefix involving\n' 
       +'random (a-z0-9) characters, used for sending listowner PIN as \n'
       +'2nd list admin authentication factor. Use lowercase letters and \n'
       +' numerals only.', 
      'help':'This email alias allows for extra admin login entropy.' },

      'owner_email':{
      'default':'person@example.org',
      'prompt':'Personal email address to which public listowner email will be redirected. ', 
      'help':'Aliasing this allows another listowner to provide holiday cover, and reduces spam.' },
      
      'UID':{
      'default':'65534',
      'prompt':'User ID of address.pkl user ownership.',
      'help':'Used when generating alias to add to /etc/aliases enabling execution of '+
             'email handling program SUID to correct group ownership '+
             'You will generally keep current value when asked e.g. for new list. '+
             'Use numeric UID for nextlist user as in /etc/passwd if exists.' },

      'GID':{
      'default':'65534',
      'prompt':'Group ID of address.pkl group ownership.',
      'help':'Used when generating alias to add to /etc/aliases enabling execution of '+
             'email handling program SUID to correct group ownership '+
             'You will generally keep current value when asked e.g. for new list. '+
             'Use numeric GID for nextlist group as in /etc/group if exists.' },

      'domain':{
      'default':'example.org',
      'prompt':'email domain of list', 
      'help':'If list posts go to testlist@example.org have example.org here' },

      'webroot':{
      'default':'http://nextlist.example.com/path/',
      'prompt':'URL of web pages before entry.cgi, unsub.cgi etc.',
      'help':'Users are given a link to this URL & program '+
             'on email footers to unsubscribe e.g.'+
             'http://nextlist.example.com/path/unsub.cgi' },

      'WhoCanPost':{
      'default':'member',
      'prompt':'WhoCanPost either email address of announcer,', 
      'help':'or member. If member all unmoderated members can post. ' },

      'UnsubscribeMethod':{
      'default':'secure',
      'prompt':'set to easy for one click unsubscription', 
      'help':'secure sends confirmation advised for small lists' },

      'Visibility':{
      'default':'private',
      'prompt':'Valid values private or public',
      'help':'private has no public subscribe page so list members'+
             'have to be added by listowner.' },

      'MaxLength':{
      'default':'1000000',
      'prompt':'Maximum length of message which list will relay in characters.',
      'help':'It is thought bad for list server & member devices to be '+
             'overloaded by accepting large attachments.' },

      'DownloadDays':{
      'default':'31',
      'prompt':'Time in days during which valid attachment downloads are held on server',
      'help':'Attachments are not relayed. If a listmember wants to download one, they'+
             'have this time after getting list email with a download link to get it from the server.' },
      } # end vars 
      } # end site_vars 
      # End of Config file : Site Variables 
      } # end cv 

def modify_config_var(filename,varname,newval):
  ''' rewrites a variable value in a
  live or testing .cfg file to configure some options likely to
  need changing.

  Parameters: filename - name of configuration file
              varname - name of variable to be substituted
              newval - new value for variable
  '''
  # regex to substitute value for config_dir
  # portability note: might need to handle windows pathnames as well as Unix
  # example of a matching line: ip4addr:1.2.3.4
  regex=r'^'+varname+r'\:.+$'
  import re
  cfh=open(filename,'r')
  cff=cfh.read() # read config file as string
  cfh.close()
  cre=re.compile(regex,re.MULTILINE)
  subtext=varname+':'+newval
  mo=cre.search(cff)
  if mo: # substitute found instances with subtext
    cff=re.sub(cre,subtext,cff)
  else:
    raise ValueError("Can't substitute invalid contents of "+filename)
  cfh=open(filename,'w')
  cfh.write(cff)
  cfh.close()

def get_listname():
  ''' prompts for valid listname '''
  valid=False
  while not valid:
    valid=True
    print('Enter listname for new list to configure or existing list to modify: ')
    listname=input('listname :')
    if listname in ['default','defaults','list']: 
      # exclude reserved words
      print('Invalid listname: '+listname+' is a reserved word')
      valid=False
      continue
    for char in listname: # a-z lowercase and _ only
        if char not in "abcdefghijklmnopqrstuvwxyz_0123456789":
          print('Listname must use lowercase letters numerals and underscores only')
          valid=False
    if valid:
        return listname

def main():
  listname=get_listname()
  cfg_directory=cgiutils.cfg_directory
  print('cfg_directory: '+cfg_directory)
  defaults_directory=os.path.join(cfg_directory,"defaults")
  print('defaults_directory: '+defaults_directory)
  if not os.path.isdir(defaults_directory):
      print('No defaults directory. Create and rerun program')
      return
  cfg_directory=os.path.join(cfg_directory,listname)
  print('cfg_directory now: '+cfg_directory)
  if not os.path.isdir(cfg_directory):
      # create directory for new listname
      os.mkdir(cfg_directory)
      print('created folder :'+cfg_directory+'for new list: '+listname)
      mime_defaults_file=os.path.join(defaults_directory,'mimetypes.txt')
      mime_listfile=os.path.join(cfg_directory,'mimetypes.txt')
      shutil.copyfile(mime_defaults_file,mime_listfile)
      print('Copied accepted mime types file to: '+mime_listfile)
      print('\nedit this file if you need to change per list policy')
  # loop once for each config file in md.cv
  for cfgf in md.cfgflist: # keep loop even though there's only 1 file for now
    filename=md.cv[cfgf]["filename"]
    full_fn=os.path.join(cfg_directory,filename) # full config filename
    if not os.path.isfile(full_fn):
      # new list, copy default file
      newconfig=True
      defaults_file=os.path.join(defaults_directory,filename)
      if not os.path.isfile(defaults_file):
        print('No defaults file: '+defaults_file+' create and rerun program')
        return
      shutil.copyfile(defaults_file,full_fn)
    else:
      newconfig=False
    # read configuration file into dictionary variable v
    v=cgiutils.get_vars(full_fn)
    # ensure list_id corresponds to listname already entered
    if 'List-Id' in v.keys():
      modify_config_var(full_fn,'List-Id',listname)
    # prompt the user for each variable in the keylist
    print('v: '+str(v)+'\n\n')
    for varn in md.cv[cfgf]["keys"]:
      cvalue=v[varn]
      print()
      print(md.cv[cfgf]["vars"][varn]['help'])
      print('\nenter d for default value: '+md.cv[cfgf]["vars"][varn]['default'])
      print('enter c for current value: '+cvalue+' or input new value')
      print(md.cv[cfgf]["vars"][varn]['prompt'])
      nvalue=input('[d, c or new value] :')
      if nvalue == 'd':
        # reset value to default
        modify_config_var(full_fn,varn,md.cv[cfgf]["vars"][varn]['default'])
      elif nvalue == '' or nvalue == 'c':
        pass # nowt to do, keep current value 
      else: # new value
        modify_config_var(full_fn,varn,nvalue)
    if cfgf == 'site_vars':
      # process additional variables derived from above configuration values
      set_derived_site_vars(full_fn,newconfig)
      print(full_fn+' file configured')
      print('Recommended you manually check and if needed tweak file '+full_fn)
    print_list_aliases(full_fn)
    print('\nThen chown -R nextlist:nextlist /usr/local/nextlist/lists/ ')
    print('Or similar to ensure listowner web access to new list folder') 


def print_list_aliases(full_fn):
  ''' works out a suitable set of aliases to add to /etc/aliases
  for new list and advises user of this program.'''
  v=cgiutils.get_vars(full_fn)
  # v should now store updated user prompted variables
  listname=v['List-Id']
  owner_email=v['owner_email']
  obscure_email=v['obscure_email']
  email_directory=cgiutils.email_directory
  emprog=f'{listname}: "| {email_directory}/nextlist.sh {listname} {v["domain"]} {v["UID"]} {v["GID"]}"'
  bounces=f'{listname}-bounces: {owner_email}' 
  obscure=f'{obscure_email}: {owner_email}' 
  owner=f'{listname}-owner: {owner_email}' 
  unsub=f'{listname}-unsubscribe: {owner_email}'
  #
  print('Recommended you copy, paste and edit these aliases for your')
  print('new list into /etc/aliases and run newaliases command.\n\n')
  for alias in [emprog,bounces,obscure,owner,unsub]:
      print(alias)

def set_derived_site_vars(full_fn,newconfig):
  ''' sets derived site variables for the config file site_vars 
  using prompted for base variables. If more than one configuration to be
  configured per list, create one such function per config file '''
  v=cgiutils.get_vars(full_fn)
  # v should now store updated user prompted variables
  listname=v['List-Id']
  webroot=v['webroot']
  oldpin=v['ownerpin']
  d={}
  d['Return-Path'] = listname+'-bounces@'+v['domain']
  d['Sender'] = listname+'-bounces@'+v['domain']
  d['List-Unsubscribe'] = webroot+'unsub.cgi'
  d['List-Subscribe'] = webroot+'sub.cgi'
  d['Listowner'] = listname+'-owner@'+v['domain']
  d['abuse'] = 'abuse@'+v['domain']
  if newconfig: # new configuration, need new secure pin
    d['ownerpin']=str(cgiutils.make_pin(
      mini=100000000000000,
      maxi=999999999999999))
  else: # can keep old pin
    d['ownerpin']=str(oldpin)
  for varn in d.keys(): 
    modify_config_var(full_fn,varn,d[varn])

if __name__ == "__main__":
  try:
      main()
  except:
    import traceback
    print("error detected in make_configure.py main()")
    traceback.print_exc()
