#!/usr/bin/env python
''' cgiutils.py
    Richard Kay, 
    Various CGI programming utilities in one package 

Copyright (C) 2002 - 2023 Richard Kay 
Email: rich (at) copsewood (dot) net

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 3 
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
'''

htmloutput=True # will be false for SMS gateway which wants text

# cfg_directory: the directory used to
# store .cfg files containing configuration variables,
# typically /etc/productname where productname is the
# name of the application using this module

# data_directory: the directory used to
# store .pkl files containing accounting, members,
# trading want/offer and security records 

import sys
import os
import validemail # now use an external validator
global_config_dir='/usr/local/nextlist/etc'
global_config_default_paths=os.path.join(global_config_dir,"default_paths.py")
if not os.access(global_config_default_paths, os.R_OK):
  print("cgiutils warning: data module location "+global_config_default_paths+" not readable.")
  sys.exit(1)
try:
  sys.path.append(global_config_dir)
  import default_paths
  cfg_directory=default_paths.cfg_dir
  data_directory=default_paths.data_dir
  web_directory=default_paths.web_dir
  email_directory=default_paths.email_dir
  # sw_directory=default_paths.sw_directory
except: 
  print('cgiutils warning: cant import default_paths') # debug
  sys.exit(1)

def is_pincookie(pinorcookie):
  ''' validates an authentication token which can be either an
  8 numeral PIN or a login number (this keys the logins table) 
  followed by an underscore followed by a 12 numeral cookie '''
  regex=r'^[1-9][0-9]{7,7}$|^[1-9][0-9]*_[1-9][0-9]{11,11}$'
  return is_valid(pinorcookie,max_leng=24,allowed=regex) 

def is_cookie(password):
  ''' validates a login number (this keys the logins table) 
  followed by an underscore followed by a 12 numeral cookie '''
  regex=r'^[1-9][0-9]*_[1-9][0-9]{11,11}$'
  return is_valid(password,max_leng=24,allowed=regex) 

def is_pin(password):
  ''' validates an 8 numeral PIN '''
  regex=r'^[1-9][0-9]{7,7}$'
  return is_valid(password,max_leng=24,allowed=regex) 

def is_url(url):
  ''' validates an input URL. The null value 'None' is treated as false. '''
  import urllib.parse
  # check for a null url value
  if url.lower() == 'none':
    return False,'null url' # treat this as null, not a local path
  # check if url has a scheme, e.g. http: or https:
  if url.find('://') == -1 : # no scheme, so prepend a default one
    url='http://'+url  
  try:
    urltuple=urllib.parse.urlparse(url,'http')
  except:
    return False,'URL unparseable by Python urlparse module'
  if not is_valid(urltuple[0], max_leng=5,allowed='^http$|^https$|^ftp$'):
    return False,'invalid scheme' # invalid scheme 
  if not is_valid(urltuple[1], max_leng=70,allowed=
      '^([A-Za-z0-9_-]+\.)+[A-Za-z0-9_-]+$'):
    return False,'invalid url domain name format'
  import socket
  try:
    ipaddr=socket.gethostbyname(urltuple[1])
  except:
    return False,'invalid url domain name, no DNS record'
  if not is_valid(urltuple[2], max_leng=70,allowed='^[\w\d\/\~\-\.\+]*$'):
    print(False,'invalid url path') 
  return True,'' 

def url_strip(url):
  ''' strips a URL, returning scheme, domain and path. Query string
      parameters are removed, in order to render URL safe from  future 
      XSS or CSRF attacks, e.g. if normal cookies as opposed to
      hidden field cookies used. is_url should be called first, to
      do basic validation. If it validates, then use this function to
      make the url safer. '''
  # check if url has a scheme, e.g. http: or https, and add one if not
  if url.find('://') == -1 : # no scheme, so prepend a default one
    url='http://'+url  
  import urllib.parse
  urlparts=list(urllib.parse.urlparse(url,'http'))
  urlparts[3]=urlparts[4]=urlparts[5]='' # we only want first 3 parts
  return urllib.parse.urlunparse(urlparts) # return stripped url as string

def login(account,password,logintype):
  ''' wraps services provided by authent.py module to provide
      a login session object to various cgi programs.
      Password is either a PIN (to login a new 
      session) or a cookie (to authenticate an existing session).
      Methods on the session object include login(), logout() and
      a cookie attribute is obtainable. 
  '''
  import authent
  import tablcgi
  from table_details import mtab
  site_vars=get_vars('site.cfg')
  dl=site_vars['data_location']
  members=tablcgi.table(mtab,dir=dl)
  sessionobject=authent.Auth(members,"pin",account,password,logintype)
  return sessionobject

def auth(session,ac_id,action,details,whom="admin",auditonly=False):
  ''' wraps authorisation service provided by authent.py '''
  import authent
  return authent.authorised(session,ac_id,action,details,whom,auditonly)

def get_option(ob,buttons=[]):
  ''' This utility function performs services common to a number of 
  .cgi scripts:
   
  ob is web input object returned by webinput.Webinput
  1. Obtains value of submit button pressed, validated through 
  list provided by caller as buttons list parameter.
  2. Outputs HTML ending error message if invalid option
 
  Return value is a tuple containing valid (True or False) 
  and option as string.
  ''' 
  # which submit button was pressed ?
  option="none"
  for word in buttons:
    if ob.has_key(word):
      option=word
      break
  if option=="none":
    html_end(False,error="Invalid submit button value.")
    return False,'none'
  return True,option

def get_hidden(ob):
  ''' Checks availability of the 3 hidden authentication fields common to 
  many .cgi scripts and validates these. 
  ob is web input object returned by webinput.Webinput
  Script returns tuple containing in order:
  1. Validity True or False boolean
  2. ac_id - account ID string 
  3. ua - user or admin login option
  4. pinentry - value of PIN entered
  '''
  session=False # no session established yet 
  if len(ob.keys()) == 0:
    # if no keys authentication data missing
    html_end(session,error="Missing authentication details.")
    return False,'','',''
  elif not ob.has_required(["ac_id","pin","ua"]):
    html_end(session,error="Missing account identifier or PIN.")
    return False,'','',''
  # account ID
  ac_id=ob.firstval("ac_id")
  if not is_valid(ac_id,max_leng=8,allowed='^[\w]+$'):
    html_end(session,error="Invalid user account id/no.")
    return False,'','',''
  # PIN entered
  pinentry=ob.firstval("pin")
  if not is_pincookie(pinentry):
    html_end(session,error="Invalid PIN format.")
    return False,'','',''
  # user/admin option
  ua=ob.firstval("ua",default="user")
  if not ua in ["user","admin"]:
    html_header(title="LETS Accounting and Directory System")
    html_end(session,error="Incorrect user/admin option entered.")
    return False,'','',''
  return True,ac_id,pinentry,ua 

def get_color(user_status,default_color='"#FFFFFF"'):
  ''' returns a bgcolor attribute related to user status
      which must be Y for active, N for inactive and P for
      provisional ''' 
  if user_status not in ['Y','N','P']:
    raise ValueError("invalid user status, must be Y, N or P")
  site_vars=get_vars('site.cfg')
  if site_vars['color_users'] == 'N':
    return default_color
  if user_status == 'Y':
    return site_vars['color_active']
  elif user_status == 'N':
    return site_vars['color_inactive']
  elif user_status == 'P':
    return site_vars['color_provisional']

def print_colortable(align='center'):
  ''' prints color table guide showing colors highlighting 
      active, inactive and provisional user statuses '''
  v=get_vars('site.cfg')
  if v['color_users'] == 'Y': # only print guide if needed
    print('<table align="'+align+'" > <tr>')
    print('<td bgcolor='+v['color_provisional']+'>provisional</td>')
    print('<td bgcolor='+v['color_active']+'>active</td>')
    print('<td bgcolor='+v['color_inactive']+'>inactive</td>')
    print('</tr> </table>')

def get_handleopt():
  ''' returns handle option field for application form
  if system uses alphabetic account identifiers. For systems
  with numeric account identifiers a comment string is returned.'''
  newm_cfgs=get_vars("newmember.cfg")
  if newm_cfgs["system_has_numeric_accounts"] == 'Y':
    formpart="<!-- form field for handle preference not needed. -->"
  else:
    formpart='''
<p> <b>Account identifier preference: </b>
On systems using initials, short nicknames or handles
for account identifiers you may suggest preferred account name.
We suggest you provide a few options with spaces between words, in
case one or some of these are already used. Please keep your
preferences short and easy to remember.
<p>
<INPUT TYPE="text" NAME="handle" SIZE="30" MAXLENGTH="40">
    '''
  return formpart

def list_ackfiles():
  ''' returns list of acknowledgement filenames including current
  file plus archived files. '''
  site_vars=get_vars('site.cfg')
  dl=site_vars['data_location']
  from table_details import atab 
  import os,os.path
  ackfnames=[]
  prefix='acks'
  bfile=os.path.join(dl,atab.file)
  if os.path.isfile(bfile):
    ackfnames.append(os.path.basename(bfile))
  # list of number of ack generations kept
  generations=list(range(int(site_vars['ack_generations'])))
  for gen in generations:
    thisfile=prefix+'.'+str(gen)+'.pkl'
    if os.path.isfile(os.path.join(dl,thisfile)):
      ackfnames.append(thisfile)
  return ackfnames
 
def make_pin(mini=100000,maxi=999999):
  ''' generates a random PIN number or cookie used for user 
  authentication or session tracking. Uses /dev/urandom if this 
  exists (Linux) in preference to the less secure Python standard 
  library random module. ''' 
  import os.path
  # check for sane range between min and max to avoid infinite loop
  if maxi < 10 or mini < 1 or (maxi - mini)/float(abs(maxi)) < 0.5:
    raise ValueError("Too small or narrow or negative range for PIN")
  # check if exception raised when using os.urandom()
  try:
    bytes=os.urandom(1)
    hasurandom=True
  except:
    hasurandom=False
  if not hasurandom: 
    import random
    return random.randrange(mini,maxi)
  else:
    # using a kernel random device should give better entropy
    digits=len(str(maxi)) # number of decimal digits in max PIN
    while(True): # loop until we generate and return a PIN between min and max
      # read as many random bytes as max has digits
      import random
      pin=random.randint(mini,maxi)
      # if random pin is in range return it, otherwise loop 
      if mini < pin < maxi: 
        return int(pin) 

def keys(ob):
  """ shortcut to form.keys() method """
  return list(ob.keys())

def has_key(ob,key):
  """ shortcut to form.has_keys() method """
  keylist=list(ob.keys)
  return key in keylist

def has_required(ob,required):
  """ checks if required keys are present in form.getvalue
  user must supply required as a list of key names, e.g.
  if not has_required(["Email","Name"]):
    cgiutils.html_end(session,error="You havn't input all required values")
  """
  return ob.has_required(required)

def firstval(ob,key,default=""):
  """ returns value from form as a single string, or default for 
  empty/wrong type. Returns first of a multi-valued submission. """
  value=ob.form.getfirst(key)
  if type(value) == type([]): # multi value submission
    if value == []: # empty list ??
      return default 
    elif type(value[0]) == type(""): # first element is string
      return value[0]
    else: # list, but 1st element not a string 
      return default 
  elif type(value) == type(""):
    return value
  else: # dont know what type of data this is
    return default 

'''
def listval(ob,key,default=[]):
  """ returns val as a list of strings"""
  form=ob.form
  value=form.getvalue(key)
  strings=[]
  if type(value) == type([]): # multi value submission
    for i in range(len(value)):
      if type(value[i]) == type(""):
        strings.append(value[i])
  elif type(value) == type(""):
    strings.append(value)
  return strings
'''

def stof(ob,key,default=None):
  """ string to float conversion or returns default for non float value"""
  string=ob.firstval(key) # converts to single string or ""
  try:
    f=float(string)
  except:
    return default 
  else:
    return f

def stoi(ob,key,default=None):
  """ string to int conversion or returns default for non int value"""
  string=ob.firstval(key) # converts to single string or ""
  try:
    i=int(string)
  except:
    return default 
  else:
    return i 

def mystr(thing):
  ''' intended to sort dict objects before stringing them '''
  if type(thing) == type(''): # nowt to do
    return thing
  elif type(thing) == type([]): # list
    return str(thing) # python str can do the right thing
  elif type(thing) == type((1,2)): # tuple 
    return str(thing) # python str can do the right thing
  elif type(thing) == type({}): # dict
    # this needs more care as there is no default ordering of a dictionary
    skeys=sorted(thing.keys()) 
    op="{"
    for key in skeys:
      value=str(thing[key])
      op+=str(key)+':'+str(value)+','
    op=op[:-1] # strip trailing comma
    op+="}"
    return op 
  else: # int or float ?
    return str(thing) 

def is_valid(data,max_leng=30,allowed=r"^\w+$",prohibited=None):
   """data (string) is valid if <= max_leng,
     AND if an allowed RE is specified:  must match allowed.
     AND must not match a prohibited RE 
     Defaults: max_leng: 30, allowed: string of 1 or more \w word chars"""
   import re
   if len(data) > max_leng:
       return 0 # not valid
   if allowed:
       match_allowed=re.search(allowed,data)
       if not match_allowed:
           return 0 # not valid
   if prohibited: # function can also be used to exclude prohibited RE
       match_prohibited=re.search(prohibited,data)
       if match_prohibited:
           return 0 # not valid
   return 1 # passed all tests so should be valid

def normalise_email(address,deliv_check=False):
    if not validemail.is_email(address,check_deliv=True):
      raise ValueError("Email address not validated or not deliverable so can't normalise it")
    # return normalised address
    return validemail.normalise_email(address,check_deliv=True)

def is_alias(alias):
    """ checks if alias is locally deliverable by being in /etc/aliases """
    aliases=open('/etc/aliases').readlines()
    prefixes=[]
    for aline in aliases:
      if aline and aline[0] != '#' and ':' in aline:
        prefix=aline.split(':')[0]
        if alias == prefix:
          return True # found it so must be valid
    return False # not found

def is_email(address,deliv_check=False):
    """ Checks whether a supplied email address is valid or not. """
    return validemail.is_email(address,check_deliv=deliv_check)

def is_person(name):
    """ Checks whether a supplied user person name is valid or not. 
    Valid names include 
    Mr. Smith
    Pat
    John Whitney-Houston
    Sean O'Heany
    Prohibited characters include the following due to special use in 
    email addresses: <>@
    """
    return is_valid(name,max_leng=100,
       allowed=r"^\w[\w '-.]+$",prohibited=r'><@')

def is_ipv4addr(address):
    """ Checks whether a supplied dot quad IPV4 address is valid or not. 
        Valid example: 12.134.67.98
        invalid include octets > 255 or < 0 .
    """
    if type(address) != type("1.2.3.4"):
      return False # not a string
    octets=address.split(".")
    if len(octets) != 4:
      return False # doesnt have 4 parts
    for octet in octets:
      try:
        val=int(octet)
      except:
        return False # not an integer value
      if not 0 <= val <= 255:
        return False # out of range
      if not 1 <= len(octet) <= 3:
        return False # float ?
      if '.' in octet:
        return False # float 
    return True # passes all tests 

def is_ipv6addr(address):
    """ Checks whether a supplied IPV6 address is valid or not. 
        Valid example: 2001:470:1f09:1ac2::10 
        invalid include chars not in range 0-9a-f: """ 
    return is_valid(address,max_leng=39,
       allowed=r'^[a-fA-F0-9\:]+$')

def make_clean(string,allowed="""abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ '-_""",
    prohibited=r"[^\w \.\,\;\:\!\$\%\-\+\*\=\#\~\/\!\n\t\?\@\(\)\\]"):
    """ cleans up string by removing prohibited characters defined by a
      regular expression character class. If prohibited == None, then
      pass through only characters in allowed string e.g. for cleaning up person name."""
    import re
    if prohibited: # remove prohibitions and ignore what's allowed
      reobj=re.compile(prohibited)
      string=reobj.sub('',string) # remove all prohibited chars 
      return string
    else:
      outp=''
      for char in string:
        if char in allowed:
          outp+=char
      return outp

def html_header(title="CGI generated HTML",
                bgcolor='"#FFFFFF"',
		cookie=None):
    """ prints HTTP/HTML header to browser  """
    if htmloutput:
      print("Content-type: text/html\n")
    else:
      print("Content-type: text/plain\n")
    import http.cookies
    if cookie:
      c=http.cookies.SimpleCookie()
      if type(cookie) == type('string'):
        print("Set-Cookie: session=%s\n" % cookie)
      elif type(cookie) == type(c):
        print(cookie.output())
      else:
        raise ValueError("cgiutils.html_header: invalid cookie type") 
    if htmloutput:
      print("<HTML><Head><Title>",title,"</Title></Head>")
      print("<Body bgcolor=",bgcolor,"><H1>",title,"</H1>")

def send_mail(fromaddress,toaddress,message,debug=0):
    """ sends a mail message """
    import smtplib
    if '@' not in toaddress:
        if not is_alias(toaddress):
            raise ValueError('unroutable alias')
    elif not is_email(toaddress,deliv_check=True):
            raise ValueError('unroutable address')
    server = smtplib.SMTP('localhost') # create server object
    if debug: server.set_debuglevel(1)
    server.sendmail(fromaddress, toaddress, message) # send message
    server.quit() # close mail server connection

def html_end(session=False,error="",received=0,want_form=0,back=0,vars=[]):
    """ ends an html output to client with assurance 
        or error message and link. 
    """
    # print(f"<p> debug: {back} {str(vars)} </p>") 
    if htmloutput:
      if error: # only happens if this parameter is filled in
        print("<p> ",error,"</p>") # print details of the error
      elif received: # standard response if valid data received and processed 
        print("<p>Thanks for your input which has been received and ")
        print("processed.</p>")
      if back == 1: # 
        print("<p>Pressing browser <b>back</b> button returns to the form.</p>")
      if type(back) == type('string'):
        print(send_form(back,vars=vars))
      print("</body></html>") # end html
    else:
      if error: # only happens if this parameter is filled in
        print("error message: "+error) # print details of the error

def send_form(html_file,vars=[],ua='user',form_dir='forms'):
    """ returns an includes processed html form to caller

    Name of file containing the form HTML is in html_file parameter.
    This file may contain %d, %f and %s and other Python string escapes.
    These escapes should match in number, type and sequence with the
    vars[] list of (typically hidden field) values to be interpolated.
    form_dir is the directory containing forms.

    In forms where additional admin options exist these are not
    presented to the user. If the form file is called e.g. the.form 
    these admin include files will be named the.form.a1 the.form.a2 etc.

    In this situation the vars list will contain None values in
    the positions where admin includes a1, a2 etc are to be used
    instead of the values provided through the vars list. So
    e.g. a vars list of [1,None,'string',None] will involve
    includes a1 and a2 being read and included instead of the
    None values. If ua == 'user' empty '' strings will be used
    instead of the include files.
    """
    if ua not in ["user","admin"]:
      raise ValueError('send_form: invalid ua option') 
    import os.path
    formname=os.path.join(form_dir,html_file)
    if not os.path.isfile(formname):
      raise IOError('form file '+formname+' does not exist') 
    inp = open(formname,'r')
    form=inp.read() # read HTML form from html_file
    inp.close()
    # process admin includes
    includeno=1 # number of the include
    ovars=[] 
    for var in vars:
      if type(var) == type(None):
        if ua == 'admin': # compute next include filename e.g. main.form.a1
          includename=formname+'.a'+str(includeno)   
          if not os.path.isfile(includename):
            raise IOError('include file '+includename+' does not exist') 
          inp = open(includename,'r')
          var=inp.read() # read HTML form inclusion admin part 
          includeno+=1
        else:
          var='' # user doesnt see an admin form option
      ovars.append(var) # seems easier than modifying vars in place
    if ovars:
      tvars=tuple(ovars) # convert ovars form variables list to tuple
      return form % tvars  # send session coded form to browser
    else: # form is constant, no variables or includes
      return form

def make_select(cat_table,
                instruction="Please select one from the following options:",
                cgiparameter="category",
                default_value=""):
  ''' creates an HTML select paragraph to string interpolate within an
      HTML form. 

      cat_table is the cgitable object containing
      category and desc keys for select categories and descriptions.
      
      cgiparameter is the name of the parameter to be passed to the
      CGI program which accepts input from the form.
'''
  sstring='<SELECT NAME="'+cgiparameter+'">\n'
  sstring+='<OPTION VALUE="'+default_value+'">' + instruction + '\n' 
  for row in cat_table.data:
    sstring+='<OPTION VALUE="' + row["category"] + '">' + row["desc"] + '\n'
  sstring+='</SELECT>\n'
  return sstring 

def make_file_select(filelist,
      instruction="Please select one from the following options:",
      cgiparameter="category", default_value=""):
  ''' Creates an HTML select paragraph to string interpolate
      within an HTML form used to select a filename from the filelist
      provided. Wraps the more general make_select function for
      the situation where a list of files can be selected between.  
  '''
  import tablcgi  
  from table_details import cattab # useful one to clone
  # clone a auditfilenames categories type table
  fopt=tablcgi.table_def(cattab)
  fopt.file=None # in memory only - don't save
  fopts=tablcgi.table(fopt)
  i=0
  for filen in filelist:
    i+=1
    frow={"sort":i,"category":filen,"desc":filen}
    fopts.addrow(frow)
  html_select_statement=make_select(fopts,
                instruction=instruction,
                cgiparameter=cgiparameter,
                default_value=default_value)
  return html_select_statement

def stripwhite(s):
  ''' removes whitespace at start and end of string '''
  while( s and s[0] in [' ','\n','\t','\r']):
    s=s[1:]
  while( s and s[-1] in [' ','\n','\t','\r']):
    s=s[:-1]
  return s

def get_vars(filename,cfgdir=cfg_directory,datadir=data_directory,swdir='.'):
  ''' reads config variables, from a config file. 
      The config file must have lines as in the following
      2 example lines:

key:value 
# comment
      
      A dictionary is returned containing key,value pairs
      as strings. lines starting with # are ignored, so are 
      usable as comments. The benefit is that site configuration 
      variables can be installed using editable whatever.cfg files to 
      minimise installers having to edit source files.

      if cfg_directory provided as None or False the file
      will be opened in current directory, otherwise it
      will be opened in the cfgdir based on the parameter provided.

      if data_directory is provided as other than None, False, "", 0 etc,
      any value for key: data_location is overridden by this parameter.
  '''
  import os.path
  if not cfg_directory:
    fh=open(filename) # open configuration file
  else:
    fh=open(os.path.join(cfgdir,filename)) # open configuration file
  vars={}
  lines=fh.readlines()
  fh.close()
  for line in lines:
    if line[0] == '#' or line[0] == ' ' or line[0] == '\t' or line[0] == '\n':
      continue # skip comments or empty lines
    name,value=line.split(':',1)
    vars[name]=stripwhite(value)
  if datadir:
    vars['data_location']=datadir
  if swdir:
    vars['sw_directory']=swdir
  fh.close()
  return vars

def tester():
  """ unit test code for module cgiutils.py """
  while 1:
    print("enter   to test")
    print("  1     data validation")
    print("  2     Send mail message")
    print("  3     Normal HTML")
    print("  4     Error HTML")
    print("  5     Form HTML")
    print("  6     validate ip4 dot quad address")
    print("  7     validate email address")
    print("  8     stringify objects")
    print("  10     Quit")
    test=int(input("option: "))
    if test == 1:
        data=input("enter data to be validated as a string: ")
        max=int(input("enter max data length as an int: "))
        allowed=input("enter allowed Regular Expression: ")
        prohibited=input("enter prohibited RE or None : ")
        if prohibited == "None": 
          prohibited=None # null object
        if is_valid(data,max,allowed,prohibited):
            print("data is valid")
        else:
            print("invalid data detected")
    elif test == 2:
        message=input("input send_mail() test message: ")    
        toaddress=input("enter to address for test message: ")
        fromaddress=input("enter from address for test message: ")
        if is_email(fromaddress) and is_email(toaddress):
            send_mail(fromaddress,toaddress,message)
            print("message sent")
        else:
            print("one or both addresses were invalid")
    elif test == 3:
        html_header(title="Testing cgiutils normal HTML",bgcolor='"#FFFF88"')
        html_end(False)
    elif test == 4:
        error=input("input error message") 
        html_header(title="Testing cgiutils error HTML",bgcolor='"#FF88FF"')
        html_end(False,error)
    elif test == 5:
        # the form facility requires a custom HTML file and variable data for a
        # secure form driven application. 
        # PIN and ID fields stay associated with a user session by putting
        # these into hidden fields.
        html_header(title="Testing cgiutils form HTML",bgcolor='"#88FFFF"')
        scratch=input("enter name for temporary scratch file e.g. temp123 : ")
        scratch='/tmp/'+scratch
        id=input("enter value for account: ")
        import random
        pin=int(input("enter (int) PIN or 0 for a random one: "))
        if not pin: pin=make_pin()
        form="""<form action="http://example.com/random/random.cgi" 
            method="post">
            <INPUT TYPE="hidden" NAME="id" VALUE="%s" >
            <INPUT TYPE="hidden" NAME="pin" VALUE="%d" >
            <p><b>View Wants</b> :
            <INPUT TYPE=Checkbox NAME="view"><p>
            <INPUT TYPE="submit"></form></body></html>"""
        sfile=open(scratch,"w") # write HTML form to scratch file
        sfile.write(form)
        sfile.close()
        print(send_form(scratch,vars=[id,pin])) # test send_form function
    elif test == 6:
        ipaddr=input("input test IPV4 address e.g. 1.2.3.4 : ") 
        if is_ipv4addr(ipaddr):
          print(ipaddr+" is valid")
        else: 
          print(ipaddr+" is not valid")
    elif test == 7:
        email=input("input test email address e.g. joe@example.net : ")
        delivcheck=input("Y if delivery check required")
        if delivcheck in "yY":
          if is_email(email,deliv_check=True):
            print(email+" is valid and passes deliverable check")
            print("normalised address: "+normalise_email(email,deliv_check=True))
          else: 
            print(email+" is not valid")
            next
        else:
          if is_email(email):
            print(email+" is valid, delivery check not performed")
            print("normalised address: "+normalise_email(email,deliv_check=False))
          else: 
            print(email+" is not valid")
            next
    elif test == 8:
        print("int: "+mystr(42))
        print("float: "+mystr(23.45))
        print("string: "+mystr("this and That"))
        print("list: "+mystr([45,"thing",98.3]))
        print("dictionary sort: "+mystr( {'who':'what','beeny':5,'any':'z','ex':'asdklfj'} ))
    elif test == 10:
        break
    else:
        print("invalid option. Try again\n")
    input("press enter to continue testing")

if __name__ == '__main__': # true when run, false when imported
    tester()

