#!/usr/bin/env python

''' authent.py provides authentication and security
    logging services
'''

# change next line False to True in order to debug

debug=False

# read constants and policy data from secure.cfg file 

import cgiutils
import tablcgi
v=cgiutils.get_vars('secure.cfg')

site_vars=cgiutils.get_vars('site.cfg')
dl=site_vars['data_location']

# badtries_per_ip determines how many invalid login
# attempts are allowed for an IP address before further
# attempts from the same address are ignored . 
# badtries_per_ac determines same per account.  If, for
# a particular time period, bad logins - good logins > badtries
# allowed then IP address or account is locked. 

badtries_per_ip={"hour":int(v['biph']),
                 "day":int(v['bipd']),
                 "week":int(v['bipw']),
                 "month":int(v['bipm']),
                 "year":int(v['bipy'])   }

badtries_per_ac={"hour":int(v['bach']),
                 "day":int(v['bacd']),
                 "week":int(v['bacw']),
                 "month":int(v['bacm']),
                 "year":int(v['bacy'])   }

log_generations=int(v['log_generations']) # no. generations of logfiles to keep

class Seconds:
  pass # does nothing but provide a namespace class

secs=Seconds()
secs.hour=3600 # all times are in seconds
secs.day=secs.hour*24
secs.week=secs.day*7
secs.month=secs.day*30 # for security purposes all months have 30 days
secs.year=secs.day*365 # for security purposes all years have 365 days 

# Log and good/bad login attempt table maintenance control

rebuild_interval = secs.day # good/bad login attempts table rebuild interval  
rotate_interval = secs.month # login log rotation interval  

# session_duration_time is number of seconds within which
# a login using the same account id and IP address is 
# considered part of the current session. A session is logged once only,
# not every time an authentication event occurs.

session_duration_time=secs.hour

# end of constants and policy data 

# If you need to edit anything below this please let me know
# rich at copsewood dot net

def authorised(session,ac_id,action,details,whom="admin",auditonly=False):
  ''' Matches who (ac_id) is doing what (action) for whom
  against authorisations table. If not authorised
  returns false, else authorised returns true. Calls
  audit function to log attempt in either case.
  
  ac_id - who is doing the action
  action - what is being done, 1 of a list of codes in authcodes.py
  whom - whom it is being done for or on behalf of
  details - description of action being carried out
  auditonly - if True, any authenticated user can do this, but
              audit the action anyway.
  '''
  
  import tablcgi
  import authcodes
  if action not in authcodes.codes():
    error=''' cgiutils.auth() detected caller coding error. Unknown
              action: '''+action+''' Please inform webmaster.'''
    cgiutils.html_end(session,error)
    return False # coding error
  from table_details import authtab
  auths=tablcgi.table(authtab,dir=dl)
  # if logged in as admin then we can act on behalf of anyone, so we
  # look for wildcard admin authorisation. Otherwise the authorisation
  # required is specific to account whom action is carried out on
  # behalf of.
   
  if ac_id == "admin":
    behalf="admin"
  else:
    behalf=whom
  # if admin has superpowers, specific authorisation is not needed 
  if ac_id == "admin" and v['admin_has_superpowers'] == 'yes':
    if not audit(session,ac_id,action,'yes',details,whom):
      print('<p> Database error try again later <p>')
      # don't authorise what can't be audited
      return False
    return True
  # if anyone can do the action, it doesn't need further authorisation.
  if auditonly: 
    if not audit(session,ac_id,action,'yes',details,whom):
      print('<p> Database error try again later <p>')
      # but don't authorise what can't be audited
      return False
    return True
  specific_key=action+','+ac_id+','+behalf
  general_key=action+','+ac_id+',admin'
  if (specific_key not in auths.keys()) and (general_key not in auths.keys()):
    audit(session,ac_id,action,'no',details,whom)
    error='Unauthorised attempt to: '+action+' by account: '+ac_id+''' logged. 
           If this was a mistake, logging it can help to identify possible 
           user interface, documentation and system improvements. If 
           you think you should be authorised to carry out this action 
           please contact the system administrator.'''
    cgiutils.html_end(session,error)
    return False
  else: # if either the specific or general key exists, the action is authorised
    if not audit(session,ac_id,action,'yes',details,whom):
      print('<p> Database error try again later <p>')
      # but don't authorise action if it can't be audited
      return False
    return True

def audit(session,ac_id,action,authorised,details,whom="admin"):
  ''' adds an audit record in respect of an administrative action
  This function is careful not to compromise PIN security by
  replacing PINs in details audited prior to storage. '''
  import tablcgi
  from table_details import aactions
  if authorised not in ['yes','no']:
    print("authent.audit: coding error in caller")
    raise ValueError
  # obscure PIN numbers from details
  import re
  rex=r"\'pin\'\:\s?\d{8,8}"
  cre=re.compile(rex,re.I)
  subtext=r"'pin': xxxxxxxx" # replace 8 digit PIN with xxxxxxxx
  mo=cre.search(details)
  while mo: # substitute all found pin tag strings with subtext
    details=re.sub(cre,subtext,details)
    mo=cre.search(details)
  # 
  aact=tablcgi.sq_table(aactions,dir=dl)
  try:
    aaid=aact.get_nextnum()
  except:
    error="""Couldn't obtain next admin audit_id. File locking problem ?
    Try again in a few minutes and if you have the same problem
    repeatedly please inform the webmaster."""
    cgiutils.html_end(session,error)
    return False # failure
  import time
  now=int(time.time())
  eventrow={"aaid":aaid,
        "ac_id":ac_id,
        "action":action,
        "when":now,
        "whom":whom,
        "auth":authorised,
        "details":details }
  aact.addrow(eventrow)
  return True # success

def pin_or_cookie(password):
  ''' checks format of password and returns one of: 
  "pin" if it is an 8 digit pin or 
  "cookie" if it is a cookie in format loginnumber_12numericcookie
  e.g. 1234_123456789012 which keys logins table with 1234
  or "none" if invalid
  '''
  if not cgiutils.is_pincookie(password):
    return 'none' 
  elif cgiutils.is_cookie(password):
    return 'cookie'
  else:
    return 'pin'

class Auth:
  '''  
    class Auth is used for operations on authentication
    objects. An Auth object is initialised with the table 
    containing the account numbers and passwords, the column
    names for accounts and PINs and the account ID for
    the user to be authenticated. Table containing
    account numbers must be keyed on account number.
    Methods provide login, logging and log checking 
    services to enable brute-force password guessing
    to be detected.
  '''

  def __init__(self,table,pw_col,account,password,logintype):
    ''' initialises an Auth (entication) object. table is
        the tablcgi table containing a/c names and passwords.
        pw_col is the key for the password column in this table. 
        account is the value for the account column which must 
        key this table. logintype is user or admin '''
    from table_details import logins 
    self.logs=tablcgi.sq_table(logins,dir=dl)
    self.table=table
    self.account=account
    self.pwinput=str(password) 
    # record time login object created 
    import time
    logtime=int(time.time())
    self.when=logtime
    self.loggedin=False
    self.expired=False
    # authentication failures for a good reason are not counted 
    self.goodreason=False 
    self.logintype=logintype # valid values are 'user' and 'admin'
    if self.logintype not in ['user','admin']:
      raise ValueError("authent.py","logintype must be user or admin")
    # set porc, newlogin, login_id and cookie parameters
    self.porc=pin_or_cookie(self.pwinput)
    if self.porc == 'pin': # new session
      self.newlogin=True
      self.login_id='' # not yet established
      self.cookie='' 
      # a new one is constructed by the log method, or obtained from
      # logins table if an existing session
    elif self.porc == 'cookie':
      self.newlogin=False
      self.login_id=self.pwinput.split('_')[0]
      self.cookie=self.pwinput
    else:
      raise ValueError("authent.py","password is neither a PIN nor a cookie")
    # check user has account 
    self.account_row=self.table.find(account)
    self.reason="N/A" # reason login might be rejected to be logged
    if self.account_row == -1:
      self.ac_exists=False
      self.authenticated=False
      self.password="" # No correct password as account does not exist
      self.reason="account does not exist" 
    else:
      self.ac_exists=True
      if self.newlogin :
        # can only do this next bit for pin input (new session) 
        self.password=str(table.data[self.account_row][pw_col])
        self.status=table.data[self.account_row]["active"]
        if ( self.password == self.pwinput and self.status in ['Y','P'] ) : 
          #  correct PIN, new login, active and provisional are OK
          self.authenticated=True # so far, further checks needed
        else:
          self.authenticated=False # unconditionally 
          self.reason="incorrect password."
          if self.status == 'N':
            self.reason="inactive account."
      else: 
        # do this next bit to authenticate/reject cookie
        #print self.login_id 
        logidx=self.logs.find(int(self.login_id))
        #print str(logidx) 
        if logidx >= 0 : #found a login record 
          # check if cookie value matches what it should
          if self.logs.data[logidx]["cookie"] != self.pwinput : 
            self.authenticated=False # unconditionally 
            self.reason="wrong cookie value"
          else: 
            # cookie is valid for login so check if cookie is current 
            if (self.when - self.logs.data[logidx]["when"] 
                < session_duration_time ):
              # OK cookie has correct value and is fresh
              self.authenticated=True # so far, further checks needed
            else:  
              # Cookie has expired. This is expected to occur 
              self.authenticated=False # unconditionally 
              self.expired=True  # needs more useful error report
              self.reason="expired session"
              self.goodreason=True # don't account this as a bad attempt
        else:
          # bad cookie, doesn't index login record
          self.authenticated=False # unconditionally 
          self.reason="bad cookie index no session record"
      # end do this bit for cookie
      # override validity of login/authentication if user inactive 
      self.active=table.data[self.account_row]["active"]
      if self.active == 'N': # inactive member should not login
        self.authenticated=False # unconditionally 
        self.reason="inactive account login disabled"
    # end of if/else branch concerning whether account exists or not
    # associate object with client IP address
    import os
    if 'REMOTE_ADDR' in os.environ:
      rem_addr=os.environ["REMOTE_ADDR"]
    else:
      rem_addr="127.0.0.1"
    self.ip=rem_addr
    # if account and password are good then
    # check if ip/account combination is overridden. Default is not.
    self.overridden=False # Overrides can apply to specific IP/account pairs
    self.ov_accept=None
    if self.authenticated: 
      from table_details import ovrid
      ovs=tablcgi.table(ovrid,dir=dl)
      for orow in ovs.data:
        if ( orow["ip"] == self.ip and 
             orow["ac_id"] == self.account and
             orow["expiry"] > self.when ):
          self.overridden=True
          if orow["ar"] == "accept":
            self.ov_accept=True # overridden to accept
          else: 
            self.ov_accept=False # overridden to reject
            self.authenticated=False
            self.reason="override rejects IP/address pair"
          break # if there are conflicting overrides the first wins
    # prepare to check for attack. Only do this if not overridden and
    # authenticated so far.
    self.attack=False # assume not unless proven otherwise
    # to minimise effects of an attack, if exclusion of address due to
    # too many attempts from an address occurs, don't add log
    # entries worsening the account so account can login from elsewhere
    self.log_account=True # add entries to account by default 
    if self.authenticated and not self.overridden:
      self.attack=self.is_attacker() # check for recent failed attempts
      if self.attack:
        self.authenticated=False
        # is_attacker() updates self.reason
    # check for a repeated good new login as no need to log these
    if self.authenticated and self.newlogin :
      self.newlogin=self.new_login() # it might not really be new
    # log authentication request either if it really is a new session
    # or if authentication failed for no good reason 
    # (e.g. a session timeout counts as a good reason) 
    if ( self.newlogin or 
         ( (not self.authenticated) and 
           ( not self.goodreason) 
         )
       ) :
      self.log() 

  def login(self):
    '''The login method processes a login attempt.'''
    if self.authenticated:
      self.loggedin=True
    else:
      self.loggedin=False
    return self.authenticated

  def logout(self):
    ''' achieves a logout by resetting the session cookie to
        a random value unknown to and not reported to the client. '''
    if not self.loggedin:
      raise ValueError("cannot logout before logged in")
    logidx=self.logs.find(int(self.login_id)) # find login record
    if logidx == -1: # should never happen if self.loggedin
      raise ValueError("cannot logout before logged in trap 2")
    # make a new unknown cookie
    cookierand=str(cgiutils.make_pin(mini=100000000000,maxi=999999999999))
    new_cookie=self.login_id+'_'+cookierand
    self.logs.data[logidx]["cookie"]=new_cookie
    self.logs.save() 

  def log(self):
    ''' writes a log record to the login log. Bad PINs are not logged
    directly, but are SHA hashed prior to logging. 
    This method should be called only for a genuine new session.  '''
    from table_details import logins 
    row={}
    if self.ac_exists:
      row["ac_exists"]='Y'
    else:
      row["ac_exists"]='N'
    # construct the session cookie using the index of the row to be added
    # and a 12 digit random number with an underscore delimiter 
    cookierand=str(cgiutils.make_pin(mini=100000000000,maxi=999999999999))
    nextnum=str(self.logs.get_nextnum())
    self.cookie=nextnum+'_'+cookierand 
    self.login_id=nextnum
    row["cookie"]=self.cookie
    row["when"]=self.when
    row["ip"]=self.ip
    if self.log_account:
      row["ac_id"]=self.account
    else:
      # using sacrificial goat account helps avoid DOS attack
      row["ac_id"]=v["goat"] 
    if self.authenticated: 
      row["goodlogin"]='Y'
      row["badpin"]='N/A'
    else:
      import hashlib
      bpwinput=self.pwinput.encode(encoding = 'UTF-8', errors = 'strict')
      hashpw=hashlib.sha1(bpwinput).hexdigest()
      row["goodlogin"]='N'
      row["badpin"]=hashpw[0:12] # first 12 chars of hash digest logged
    row["reason"]=self.reason
    self.logs.addrow(row) # log it
    self.update_gbips() # quick/dirty update good/bad IP login attempts 
    self.update_lacs() # quick/dirty update good/bad account login attempts 
    # in live mode this maintenance is done hourly using routine.py 
    # we make it occur here in test mode so we can test it has occurred. 
    mode=open("mode.txt").read()
    if mode[-1] == '\n':
      mode=mode[:-1]
    if mode not in ['test','live']:
      raise ValueError("invalid mode.txt value must be test or live")
    if mode == 'test':
      maintenance() 
 
  def update_gbips(self):
    ''' does a quick and dirty update of gbips table without
        ageing and timing out data in response to logging of a single 
        record. See maintenance() function for proper rebuild, but
        proper rebuild probably should occur less frequently than this
        update. '''
    from table_details import gbips
    gbip=tablcgi.table(gbips,dir=dl)
    rowpos=gbip.find(self.ip)
    if rowpos == -1: # true if IP address unknown
      if self.authenticated:
        gbrow={"ip":self.ip,"glh":1,"gld":1,"glw":1,"glm":1,"gly":1, 
                          "blh":0,"bld":0,"blw":0,"blm":0,"bly":0}
      else: 
        gbrow={"ip":self.ip,"glh":0,"gld":0,"glw":0,"glm":0,"gly":0, 
                          "blh":1,"bld":1,"blw":1,"blm":1,"bly":1}
      gbip.addrow(gbrow)
    else:
      gbrow=gbip.data[rowpos] 
      if self.authenticated:
        gbrow["glh"]+=1
        gbrow["gld"]+=1
        gbrow["glw"]+=1
        gbrow["glm"]+=1
        gbrow["gly"]+=1
      else:
        gbrow["blh"]+=1
        gbrow["bld"]+=1
        gbrow["blw"]+=1
        gbrow["blm"]+=1
        gbrow["bly"]+=1
      gbip.modrow(gbrow)


  def update_lacs(self):
    ''' does a quick and dirty update of lacs table without
        ageing and timing out data in response to logging of a single 
        record. See maintenance() function for proper rebuild, but
        proper rebuild probably should occur less frequently than this
        update. '''
    from table_details import lacs 
    from table_details import gbips
    lac=tablcgi.table(lacs,dir=dl)
    # if log_account use account row, otherwise use goat_a/c
    if not self.log_account: 
      self.account=v["goat"]
    rowpos=lac.find(self.account)
    if rowpos == -1: # true if account unknown
      if self.authenticated:
        lacrow={"ac_id":self.account,"glh":1,"gld":1,"glw":1,"glm":1,"gly":1, 
                          "blh":0,"bld":0,"blw":0,"blm":0,"bly":0}
      else: 
        lacrow={"ac_id":self.account,"glh":0,"gld":0,"glw":0,"glm":0,"gly":0, 
                          "blh":1,"bld":1,"blw":1,"blm":1,"bly":1}
      lac.addrow(lacrow)
    else:
      lacrow=lac.data[rowpos] 
      if self.authenticated:
        lacrow["glh"]+=1
        lacrow["gld"]+=1
        lacrow["glw"]+=1
        lacrow["glm"]+=1
        lacrow["gly"]+=1
      else:
        lacrow["blh"]+=1
        lacrow["bld"]+=1
        lacrow["blw"]+=1
        lacrow["blm"]+=1
        lacrow["bly"]+=1
      lac.modrow(lacrow)

  def is_attacker(self):
    ''' Checks good/bad tables for too many recent and bad attempts 
        by either a/c or ip address. '''
    from table_details import lacs 
    from table_details import gbips
    lac=tablcgi.table(lacs,dir=dl)
    gbip=tablcgi.table(gbips,dir=dl)
    growpos=gbip.find(self.ip)
    lrowpos=lac.find(self.account)
    if lrowpos == -1 and growpos == -1:
      # no previous information on either IP or address
      return False
 
    # lock login due to bad address data first in preference to bad account
    # data second so account can still login from other addresses
    if growpos >= 0:
      if (gbip.data[growpos]["blh"] - gbip.data[growpos]["glh"] 
           > badtries_per_ip["hour"]):
        self.reason="too many bad IP attempts in hour"
        self.log_account=False # log against sacrificial goat account 
        return True
      elif (gbip.data[growpos]["bld"] - gbip.data[growpos]["gld"] 
           > badtries_per_ip["day"]):
        self.reason="too many bad IP attempts in day"
        self.log_account=False # log against sacrificial goat account 
        return True
      elif (gbip.data[growpos]["blw"] - gbip.data[growpos]["glw"] 
           > badtries_per_ip["week"]):
        self.reason="too many bad IP attempts in week"
        self.log_account=False # log against sacrificial goat account 
        return True
      elif (gbip.data[growpos]["blm"] - gbip.data[growpos]["glm"] 
           > badtries_per_ip["month"]):
        self.reason="too many bad IP attempts in month"
        self.log_account=False # log against sacrificial goat account 
        return True
      elif (gbip.data[growpos]["bly"] - gbip.data[growpos]["gly"] 
           > badtries_per_ip["year"]):
        self.reason="too many bad IP attempts in year"
        self.log_account=False # log against sacrificial goat account 
        return True
    if lrowpos >= 0:
      if (lac.data[lrowpos]["blh"] - lac.data[lrowpos]["glh"] 
           > badtries_per_ac["hour"]):
        self.reason="too many bad a/c attempts in hour"
        return True
      elif (lac.data[lrowpos]["bld"] - lac.data[lrowpos]["gld"] 
           > badtries_per_ac["day"]):
        self.reason="too many bad a/c attempts in day"
        return True
      elif (lac.data[lrowpos]["blw"] - lac.data[lrowpos]["glw"] 
           > badtries_per_ac["week"]):
        self.reason="too many bad a/c attempts in week"
        return True
      elif (lac.data[lrowpos]["blm"] - lac.data[lrowpos]["glm"] 
           > badtries_per_ac["month"]):
        self.reason="too many bad a/c attempts in month"
        return True
      elif (lac.data[lrowpos]["bly"] - lac.data[lrowpos]["gly"] 
           > badtries_per_ac["year"]):
        self.reason="too many bad a/c attempts in year"
        return True
    # all tests passed OK and no indication of attack
    return False

  def new_login(self):
    ''' called to avoid repeated good logins being recorded
        as new sessions to reduce unneccessary logging.
        If it is a new login but an old session the cookie
        must be set.  ''' 
    for oldrow in self.logs.data:
      if (oldrow["ip"] == self.ip and oldrow["goodlogin"] == 'Y' and
          oldrow["ac_id"] == self.account and
          (self.when - oldrow["when"] < session_duration_time )):
        self.cookie=oldrow["cookie"] # reload it into object
        return False # found old row indicating session is current
    return True # no logged row for current session so must be a new session

def maintenance():
  ''' Performs proper rebuild of IP and account good/bad logins 
  tables, log rotation and override expiry. This is done based on 
  time elapsed since last maintenance using on times stored in the
  smaint table. 
  '''
  from table_details import smaint, lacs, gbips
  import tablcgi
  import time
  now=time.time() 
  smain=tablcgi.table(smaint,dir=dl) 
  lac=tablcgi.table(lacs,dir=dl) 
  gbip=tablcgi.table(gbips,dir=dl)
  # obtain maintenance intervals 
  if len(smain.data) == 0:
    # populate smaint table if it is empty (initial condition)
    mrow={"lrotate":now-rotate_interval,"nrotate":now,
          "lgbr":now-rebuild_interval,"ngbr":now}
    #print "authent.maintenance initialising schedule: "+str(mrow) 
    smain.addrow(mrow)
  else:
    mrow=smain.data[0]
  # rotate logs if overdue 
  if now >= mrow["nrotate"]:
    #print "authent.maintenance rotating logfiles" 
    rotate_logfiles("logins")
    rotate_logfiles("aactions")
    mrow["lrotate"]=now
    mrow["nrotate"]=now+rotate_interval
    smain.modrow(mrow)
  else:
    pass
    #print "authent.maintenance logfile maintenance not due" 
  # rebuild good/bad tables if overdue
  if now >= mrow["ngbr"]:
    #print "authent.maintenance rebuilding good/bad tables" 
    rebuild_gbtables()
    mrow["lgbr"]=now
    mrow["ngbr"]=now+rebuild_interval
    smain.modrow(mrow)
  else:
    pass
    #print "authent.maintenance rebuilding good/bad tables not due" 
  # get rid of expired override records whenever maintenance() is run
  expire_overrides()

def expire_overrides():
  ''' removes override records which have expiry dates in the past '''
  import time
  from table_details import ovrid
  ovs=tablcgi.table(ovrid,dir=dl)
  timenow=time.time()
  delrows=[] # list of keys to delete
  for orow in ovs.data:
    ovid=orow["ovid"]
    expiry=orow["expiry"]
    if expiry < timenow:
      delrows.append(ovid)
  for ov_id in delrows:
    ovs.delrow(ov_id)
  ovs.save()

def list_logfiles(ltype):
  ''' returns list of current logins or aactions 
      rotated login or audit action logfiles. This is
      currently used to enable one to be selected for viewing
      using an HTML select statement. Note these are
      listed and selected using their basenames, e.g. 
      logins.pkl as opposed to data/logins.pkl 
      or aactions.pkl as opposed to data/aactions.pkl'''
  if ltype not in ["logins","aactions"]:
    raise ValueError
  from table_details import logins,aactions 
  import os,os.path
  logfilenames=[]
  if ltype == "logins":
    bfile=os.path.join(dl,logins.file)
  else: # ltype == "aactions"
    bfile=os.path.join(dl,aactions.file)
  if os.path.isfile(bfile):
    logfilenames.append(os.path.basename(bfile))
  # list of number of log generations kept
  generations=list(range(log_generations))
  for gen in generations:
    thisfile=ltype+'.'+str(gen)+'.pkl'
    if os.path.isfile(os.path.join(dl,thisfile)):
      logfilenames.append(thisfile)
  return logfilenames

def rotate_logfiles(ltype):
  ''' rotates logins and aactions logfiles '''
  if ltype not in ["logins","aactions"]:
    raise ValueError
  from table_details import logins, aactions
  import tablcgi
  import os, os.path
  generations=list(range(log_generations))
  # list of number of log generations kept
  generations.reverse() # want to count down not up
  for gen in generations:
    thisfile=os.path.join(dl,ltype+'.'+str(gen)+'.pkl')
    nextfile=os.path.join(dl,ltype+'.'+str(gen+1)+'.pkl')
    if os.path.isfile(thisfile):
      os.rename(thisfile,nextfile)
  if ltype == "logins":
    bfile=os.path.join(dl,logins.file)
  else: # ltype == "aactions"
    bfile=os.path.join(dl,aactions.file)
  if os.path.isfile(os.path.join(dl,bfile)):
    # this one has to be concurrency protected. The older 
    # generations are read only.
    nextfile=os.path.join(dl,ltype+'.0.pkl')
    import LockFile
    lockfilename=os.path.join(dl,bfile+".lock")
    lock=LockFile.LockFile(lockfilename,withlogging=1)
    try:
      try:
        lock.lock(15)
        os.rename(os.path.join(dl,bfile),nextfile)
      finally:
        lock.unlock()
    except LockFile.LockError:
      errormes="rotate_logfiles: could not get exclusive lock on file "+bfile
      raise errormes

def rebuild_gbtables():
  ''' rebuilds good/bad ip and accounts tables from scratch using 
      current and rotated login data '''
  if debug:
    print("rebuilding good/bad tables") 
  from table_details import lacs, gbips, logins
  import tablcgi
  import os 
  lac=tablcgi.table(lacs,dir=dl) 
  gbip=tablcgi.table(gbips,dir=dl)
  # override clogins.file cloned attribute to avoid poisoning logins 
  # object referenced elsewhere
  clogins=tablcgi.table_def(logins) 
 
  # clear all old data from lacs and gbips tables
  lac.data=[]
  lac.save()
  gbip.data=[]
  gbip.save()
  import time
  now=time.time()
  # reopen tables 
  gbip=tablcgi.table(gbips,dir=dl)
  lac=tablcgi.table(lacs,dir=dl)

  # compile a list of logfile names
  logfilenames=list_logfiles("logins")
  # process all records in all logfiles
  for filen in logfilenames:
    filen=os.path.join(dl,filen)
    # override table_def object with logfilename
    clogins.file=filen 
    log=tablcgi.table(clogins,dir=dl)
    for lrow in log.data:
      ip=lrow["ip"]
      account=lrow["ac_id"]
      age=now - lrow["when"]
      if age > secs.year :
        continue 
        # ignore all outdated records, skip to next
      # do update of gbip and lac rows. First reopen tables.
      irowpos=gbip.find(ip)
      arowpos=lac.find(account)
      if irowpos == -1: # true if IP address unknown
        # initialise whole new row null, just increment fields as needed
        gbrow={"ip":ip,"glh":0,"gld":0,"glw":0,"glm":0,"gly":0, 
                          "blh":0,"bld":0,"blw":0,"blm":0,"bly":0}
      else:
        gbrow=gbip.data[irowpos]
      if arowpos == -1: # true if account unknown
        # initialise whole new row null, just increment fields as needed
        acrow={"ac_id":account,"glh":0,"gld":0,"glw":0,"glm":0,"gly":0, 
                          "blh":0,"bld":0,"blw":0,"blm":0,"bly":0}
      else:
        acrow=lac.data[arowpos]
 
      if lrow["goodlogin"] == 'Y':
        if age < secs.hour:
          gbrow["glh"]+=1
          acrow["glh"]+=1
        if age < secs.day:
          gbrow["gld"]+=1
          acrow["gld"]+=1
        if age < secs.week:
          gbrow["glw"]+=1
          acrow["glw"]+=1
        if age < secs.month:
          gbrow["glm"]+=1
          acrow["glm"]+=1
        if age < secs.year:
          gbrow["gly"]+=1
          acrow["gly"]+=1
      else:
        if age < secs.hour:
          gbrow["blh"]+=1
          acrow["blh"]+=1
        if age < secs.day:
          gbrow["bld"]+=1
          acrow["bld"]+=1
        if age < secs.week:
          gbrow["blw"]+=1
          acrow["blw"]+=1
        if age < secs.month:
          gbrow["blm"]+=1
          acrow["blm"]+=1
        if age < secs.year:
          gbrow["bly"]+=1
          acrow["bly"]+=1
      if irowpos == -1: # true if new record 
        gbip.addrow(gbrow)
      else: 
        gbip.modrow(gbrow)
   
      if arowpos == -1: # true if new record 
        lac.addrow(acrow)
      else: 
        lac.modrow(acrow)

def get_vars():
  ''' reads the security config variables, from a config file '''
  fh=open('secure.cfg') # test configuration file
  vars={}
  lines=fh.readlines()
  fh.close()
  for line in lines:
    if line[0] == '#' or line[0] == ' ' or line[0] == '\t' or line[0] == '\n':
      continue # skip comments or empty lines
    name,value=line.split(':',1)
    if value[-1] == '\n':
      value=value[:-1] # strip newline from end
    vars[name]=value
  fh.close()
  return vars

def test():
  from table_details import mtab
  import tablcgi
  members=tablcgi.table(mtab,dir=dl)
  authobj=Auth(members,"pin","admin","12345678","admin")
  if authobj.login():
    print('successful login test passed. User: admin logged in with password: 12345678')
  else:
    print('good login test failed')
  authobj=Auth(members,"pin","admin","12345679","admin")
  if not authobj.login():
    print('wrong password test passed: admin did not log in with password: 12345679')
  else:
    print('bad login test failed')
